<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Achievement;

class AchievementController extends Controller
{
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048', // Each image should be a valid image file
        ]);

        $imagePaths = [];
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $path = $image->store('uploads', 'public'); // Save to public storage
                $imagePaths[] = $path;
            }
        }
        $post = Achievement::create([
            'title' => $request->title,
            'description' => $request->description,
            'images' => $imagePaths,
        ]);

        return response()->json(['data' => $post], 201);
    }

    public function index()
    {
        // Fetch all posts from the database
        $posts = Achievement::all();

        // Format the response data
        $response = $posts->map(function ($post) {
            return [
                'id' => $post->id,
                'title' => $post->title,
                'description' => $post->description,
                'images' => array_map(function ($imagePath) {
                    // Generate the full URL for each image
                    return url('storage/' . $imagePath);
                }, $post->images ?? []),
                'created_at' => $post->created_at,
                'updated_at' => $post->updated_at,
            ];
        });

        // Return the response as JSON
        return response()->json(['posts' => $response], 200);
    }

    public function update(Request $request, $id)
    {
        // Find the post by ID
        $post = Achievement::find($id);

        if (!$post) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        // Validate incoming request
        $request->validate([
            'title' => 'sometimes|string|max:255',
            'description' => 'sometimes|string',
            'images.*' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Update title and description if provided
        $post->title = $request->title ?? $post->title;
        $post->description = $request->description ?? $post->description;

        // Handle image uploads if any
        if ($request->hasFile('images')) {
            // Delete existing images if needed (optional, depending on your requirements)
            foreach ($post->images as $imagePath) {
                \Storage::disk('public')->delete($imagePath);
            }

            $imagePaths = [];
            foreach ($request->file('images') as $image) {
                $path = $image->store('uploads', 'public');
                $imagePaths[] = $path;
            }
            $post->images = $imagePaths; // Update images with new paths
        }

        // Save the updated post
        $post->save();

        return response()->json(['post' => $post], 200);
    }

    public function destroy($id)
    {
        // Find the post by ID
        $post = Achievement::find($id);

        if (!$post) {
            return response()->json(['error' => 'Post not found'], 404);
        }

        // Delete associated images from storage
        foreach ($post->images as $imagePath) {
            \Storage::disk('public')->delete($imagePath);
        }

        // Delete the post
        $post->delete();

        return response()->json(['message' => 'Post deleted successfully'], 200);
    }
}